/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.reporting;

import java.util.HashMap;
import java.util.Map;

/**
 * This class provides a skeletal implementation of a {@link
 * Line}.  Subclasses should override the {@link
 * #instantiateDetailPage} method if the line provides a
 * deeper detail page.
 */
public class AbstractLine implements Line {
  
  Report report;
  Page owner,detailPage;
  LineCategory category;
  Map fieldsByColumn=new HashMap();
  int state=Report.READY;

  public Page openDetailPage() {
    if (detailPage!=null) {
      throw new IllegalStateException();
    }
    detailPage=instantiateDetailPage();
    if (detailPage!=null) {
      detailPage.setMasterLine(this);
      detailPage.setReport(report);
    }
    return detailPage;
  }
  
  /**
   * Set the page that own's this line, in other words, the master page.
   */
  public void setOwner(Page owner) {
    this.owner=owner;
  }

  public Page getOwner() {
    return owner;
  }

  public void setReport(Report report) {
    this.report=report;
  }

  public void setCategory(LineCategory category) {
    this.category=category;
  }

  public Page getDetailPage() {
    return detailPage;
  }

  /**
   * Set a piece of information against this line.
   */
  public void setField(Column column,Object value) {
    assertValidColumnForCategory(column);
    if (column.getType() instanceof Object && column.isAcceptableValue(value)) {
      fieldsByColumn.put(column,value);
    } else {
      throw new IllegalArgumentException("Column "+column+" does not accept values of type "+value.getClass());
    }
  }

  public void setField(Column column,int value) {
    setField(column,new Integer(value));
  }

  public void setField(Column column,double value) {
    setField(column,new Double(value));
  }

  public void setField(Column column,boolean value) {
    setField(column,new Boolean(value));
  }

  /**
   * Subclasses should override this method if the line has
   * information in a detail page.
   */
  protected Page instantiateDetailPage() {
    return null;
  }

  /**
   * Get the field value for a given column.
   * <ul>
   * <li>ints are returned as Integer instances.
   * <li>doubles are returned as Double instances.
   * <li>booleans are returned as Boolean instances.
   * </ul>
   * @param column the given column.
   * @return null if no value has been set for the given column.
   * @throws IllegalArgumentException if the column is not known to
   * this line's category.
   */
  public Object getField(Column column) throws IllegalArgumentException {
    assertValidColumnForCategory(column);
    return fieldsByColumn.get(column);
  }

  public void close() throws ReportingException {
    state=Report.CLOSED;
    if (getDetailPage()!=null) {
      getDetailPage().close();
    }
  }

  public boolean isClosed() {
    return state==Report.CLOSED;
  }

  protected void assertValidColumnForCategory(Column column) throws IllegalArgumentException {
    if (category==null) {
      return;
    }
    if (!category.getColumns().contains(column)) {
      throw new IllegalArgumentException("Column "+column+" cannot be used here since it is unknown to this line's category, which is "+category+"("+category.getClass().getName()+")");
    }
  }

}
