/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage.reporting.collation;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import com.jcoverage.coverage.Instrumentation;
import com.jcoverage.reporting.AbstractPage;
import com.jcoverage.reporting.Line;
import com.jcoverage.reporting.LineCategory;
import com.jcoverage.reporting.Page;

/**
 * This class implements a page detailing the coverage results of a
 * single java source file.
 */
public class JavaFilePage extends AbstractPage implements Page {
  
  /**
   * We can receive multiple instrumentation contributions, since a
   * single java source file can contain multiple classes (although
   * strictly only one of these can be declared <code>public</code>).
   */
  Set instrumentations=new HashSet();

  /**
   * The line that summarizes this detail page. We prefer not to
   * duplicate information- it's either in the summary line or in this
   * page, not both.
   */
  JavaFileLine masterLine;

  /**
   * This set is important for ordering all the coverage lines as they
   * come in- the algorithm that highlights uncovered lines is
   * dependent on all the lines being in order.
   */
  SortedSet coverageUnion=new TreeSet(new Comparator() {
      public int compare(Object o1,Object o2) {
        int i1=((Integer)((Map.Entry)o1).getKey()).intValue();
        int i2=((Integer)((Map.Entry)o2).getKey()).intValue();
        if (i1<i2) {
          return -1; 
        } else {
          return 1;
        }
      }
    });

  public JavaFilePage() {
    super("File");
  }

  public Set getSourceFileLineCoverageSet() {
    return coverageUnion;
  }

  public JavaFileLine getJavaFileLine() {
    return masterLine;
  }

  /**
   * We want to know which of the lines are valid, because we don't
   * want to count lines that are impossible to reach (blank lines,
   * braces, import statements, method headers, etc..) in the coverage
   * calculations and highlighting.
   */
  Set validSourceLines=new HashSet();

  /**
   * We add instrumentation to this instance, so we can work out which lines are hit and which are not.
   */
  public void addInstrumentation(Instrumentation instrumentation) {

    instrumentations.add(instrumentation);
    coverageUnion.addAll(instrumentation.getCoverage().entrySet());

    validSourceLines.addAll(instrumentation.getSourceLineNumbers());
  }

  /**
   * This method overrides {@link
   * com.jcoverage.reporting.AbstractPage#setMasterLine(com.jcoverage.reporting.Line)} so that
   * we have convenient access to our master line, and don't have to
   * constantly cast it.
   */
  public void setMasterLine(Line masterLine) {
    super.setMasterLine(masterLine);
    this.masterLine=(JavaFileLine)masterLine;
  }

  public Set getValidSourceLines() {
    return validSourceLines;
  }

  public int getSourceLinesCount() {
    int sourceLines=0;
    for (Iterator it=instrumentations.iterator();it.hasNext();) {
      sourceLines+=((Instrumentation)it.next()).getSourceLineNumbers().size();
    }
    return sourceLines;
  }

  public double getLineCoverageRate() {
    int sourceLines=getSourceLinesCount();
    if (sourceLines==0) {
      return 0d;
    } else {
      return (double)coverageUnion.size()/sourceLines;
    }
  }

  public double getBranchCoverageRate() {
    if(getLineCoverageRate()==0d) {
      return 0d;
    }

    double total=0d;
    
    Iterator i=instrumentations.iterator();
    while(i.hasNext()) {
      total+=((Instrumentation)i.next()).getBranchCoverageRate();
    }

    return total/instrumentations.size();
  }
}
