/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage;

import com.jcoverage.util.SetHelper;

import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;


class InstrumentationImpl implements InstrumentationInternal,HasBeenInstrumented {
  static final long serialVersionUID=247748305779236308L;

  static final transient Long ZERO=new Long(0);

  final Map lineCounts=new TreeMap();
  final Set sourceLineNumbers=new TreeSet();
  final Map sourceLineNumbersByMethod=new TreeMap();
  final Map conditionalsByMethod=new TreeMap();
  final Set methodNamesAndSignatures=new TreeSet();

  int linesOfCode=0;
  String sourceFileName;


  InstrumentationImpl() {
  }

  public Map getCoverage() {
	return Collections.unmodifiableMap(lineCounts);
  }

  public long getCoverage(int lineNumber) {
	return getLineCount(lineNumber).longValue();
  }

  public void touch(int lineNumber) {
	Integer key=new Integer(lineNumber);
	setLineCount(key,increment(getLineCount(key)));
  }

  public void merge(Instrumentation instrumentation) {
	sourceLineNumbers.addAll(instrumentation.getSourceLineNumbers());
	methodNamesAndSignatures.addAll(instrumentation.getMethodNamesAndSignatures());
	sourceLineNumbersByMethod.putAll(((InstrumentationImpl)instrumentation).getSourceLineNumbersByMethod());
	conditionalsByMethod.putAll(((InstrumentationImpl)instrumentation).getConditionalsByMethod());

	Iterator i=instrumentation.getCoverage().entrySet().iterator();
	while(i.hasNext()) {
	  Map.Entry entry=(Map.Entry)i.next();

	  if(lineCounts.containsKey(entry.getKey())) {
		long total=((Long)entry.getValue()).longValue()+getLineCount((Integer)entry.getKey()).longValue();
		setLineCount((Integer)entry.getKey(),new Long(total));
	  } else {
		setLineCount((Integer)entry.getKey(),(Long)entry.getValue());
	  }
	}

	if(getSourceFileName()==null) {
	  setSourceFileName(instrumentation.getSourceFileName());
	}
  }

  public Set getSourceLineNumbers() {
	return Collections.unmodifiableSet(sourceLineNumbers);
  }

  public void setSourceLineNumbers(Set sourceLineNumbers) {
	this.sourceLineNumbers.addAll(sourceLineNumbers);
  }

  public double getLineCoverageRate() {
	if(sourceLineNumbers.size()==0) {
	  return 1d;
	}
	return ((double)lineCounts.keySet().size())/((double)sourceLineNumbers.size());
  }

  // change for djUnit
  public double getBranchCoverageRate() {
//	double sum=0d;
	double sum=1d;

	int requiredHitCount = 0;
	int hitCount = 0;

	Iterator i=sourceLineNumbersByMethod.keySet().iterator();
	while(i.hasNext()) {
//	  sum+=getBranchCoverageRate((String)i.next());
		String key = (String) i.next();
		Set requiredHits = getRequiredHits(key);
		int count = getHitCountInRequired(key, requiredHits);

		printConsole("[Required hits] " + sourceFileName + "#" + key + " - " + requiredHits);

		requiredHitCount += requiredHits.size();
		hitCount += count;
	}

	if (requiredHitCount != 0) {
		sum = (double) hitCount / (double) requiredHitCount;
	}

	return sum;

//	return sum/(double)sourceLineNumbersByMethod.keySet().size();
  }


  public double getLineCoverageRate(String methodNameAndSignature) {
	if(!sourceLineNumbersByMethod.containsKey(methodNameAndSignature)) {

	  throw new IllegalArgumentException(methodNameAndSignature);
	}

	Set lineNumbers=(Set)sourceLineNumbersByMethod.get(methodNameAndSignature);
	if(lineNumbers.size()==0) {
	  return 1d;
	}

	int count=0;
	Iterator i=lineNumbers.iterator();
	while(i.hasNext()) {
	  if(getLineCount((Integer)i.next()).longValue()>0) {
		count++;
	  }
	}

	return ((double)count)/((double)lineNumbers.size());
  }

  Set getTouchedLines(String methodNameAndSignature) {
	Set results=new TreeSet();

	Iterator i=((Set)sourceLineNumbersByMethod.get(methodNameAndSignature)).iterator();
	while(i.hasNext()) {
	  Integer lineNumber=(Integer)i.next();
	  if(getLineCount(lineNumber).longValue()>0) {
		results.add(lineNumber);
	  }
	}
	return results;
  }

  public double getBranchCoverageRate(String methodNameAndSignature) {

	if(!sourceLineNumbersByMethod.containsKey(methodNameAndSignature)) {

	  throw new IllegalArgumentException(methodNameAndSignature);
	}

	Set conditionals=(Set)conditionalsByMethod.get(methodNameAndSignature);
	if(conditionals.size()==0) {
	  // no conditional branches, therefore 100% branch coverage.
	  return 1d;
	}

	Set requiredHits=new TreeSet();
	Iterator i=conditionals.iterator();
	while(i.hasNext()) {
	  Conditional conditional=(Conditional)i.next();
	  requiredHits.add(findNextSourceLineAfter(methodNameAndSignature,conditional.getLineNumber()));
	  requiredHits.add(new Integer(conditional.getTargetLineNumber()));
	}

	return ((double)SetHelper.intersection(requiredHits,getTouchedLines(methodNameAndSignature)).size())/((double)requiredHits.size());
  }

  // add for djUnit
  private int getHitCountInRequired(String methodNameAndSignature, Set requiredHits) {
  	return SetHelper.intersection(requiredHits,getTouchedLines(methodNameAndSignature)).size();
  }

  // add for djUnit
  private Set getRequiredHits(String methodNameAndSignature) {

	Set requiredHits=new TreeSet();

  	Set conditionals=(Set)conditionalsByMethod.get(methodNameAndSignature);
  	if (conditionals.size() == 0) return requiredHits;

	Iterator i=conditionals.iterator();
	while(i.hasNext()) {
	  Conditional conditional=(Conditional)i.next();
	  int lineNumber = conditional.getLineNumber();
	  if (lineNumber >= 0) {
	  	requiredHits.add(new Integer(lineNumber));
	  }
	  Integer nextSourceAfter = findNextSourceLineAfter(methodNameAndSignature,conditional.getLineNumber());
	  if (nextSourceAfter != null && nextSourceAfter.intValue() > 0) {
	  	requiredHits.add(findNextSourceLineAfter(methodNameAndSignature,conditional.getLineNumber()));
	  }
	  int targetLine = conditional.getTargetLineNumber();
	  if (targetLine > 0) {
	  	requiredHits.add(new Integer(conditional.getTargetLineNumber()));
	  }
	}
	return requiredHits;
  }

  // change for djUnit
  Integer findNextSourceLineAfter(String methodNameAndSignature,int thisOne) {

  	if (thisOne <= 0) return new Integer(0);

	Iterator i=((Set)sourceLineNumbersByMethod.get(methodNameAndSignature)).iterator();
	Integer lineNumber=new Integer(0);

//	while(i.hasNext()&&(lineNumber.intValue()<thisOne)) {
//	  lineNumber=(Integer)i.next();
//	}
	while (i.hasNext()) {
		Integer line = (Integer) i.next();
		if (line.intValue() > thisOne) {
			lineNumber = line;
			break;
		}
	}

	return lineNumber;
  }


  public void setSourceFileName(String sourceFileName) {
	this.sourceFileName=sourceFileName;
  }

  public String getSourceFileName() {
	return sourceFileName;
  }

  private Long increment(Long count) {
	return new Long(count.longValue()+1);
  }

  private Long getLineCount(int lineNumber) {
	return getLineCount(new Integer(lineNumber));
  }

  private Long getLineCount(Integer lineNumber) {
	if(!lineCounts.containsKey(lineNumber)) {
	  lineCounts.put(lineNumber,ZERO);
	}

	return (Long)lineCounts.get(lineNumber);
  }

  private void setLineCount(Integer lineNumber,Long lineCount) {
	lineCounts.put(lineNumber,lineCount);
  }

  public Map getSourceLineNumbersByMethod() {
	return sourceLineNumbersByMethod;
  }

  public void setSourceLineNumbersByMethod(Map sourceLineNumbersByMethod) {
	this.sourceLineNumbersByMethod.putAll(sourceLineNumbersByMethod);
  }

  public Map getConditionalsByMethod() {
	return conditionalsByMethod;
  }

  public void setConditionalsByMethod(Map conditionalsByMethod) {
	this.conditionalsByMethod.putAll(conditionalsByMethod);
  }

  public Set getMethodNamesAndSignatures() {
	return methodNamesAndSignatures;
  }

  public void setMethodNamesAndSignatures(Set x) {

	this.methodNamesAndSignatures.addAll(x);

  }

  private void printConsole(Object object) {
  	String value = System.getProperty("coverage.debug");
  	if (value == null || !"true".equalsIgnoreCase(value)) return;
  	System.out.println(object);
  }

}
