/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage;

import java.io.Serializable;
import java.util.Map;
import java.util.Set;

/**
 * Instrumentation information is typically serialized to a file. An
 * instance of this class records coverage information for a single
 * class that has been instrumented.
 */
public interface Instrumentation extends Serializable {
  /**
   * default file name used to write instrumentation information.
   */
  String FILE_NAME="jcoverage.ser";

  /**
   * Increment the number of hits for a particular line of code.
   * @param lineNumber the line of code to increment the number of hits.
   */
  void touch(int lineNumber);

  /**
   * @return a sorted coverage mapping from source line to number of
   * hits.
   */
  Map getCoverage();

  /**
   * @return the number of hits a particular line of code has.
   * @param lineNumber the source code line number.
   */
  long getCoverage(int lineNumber);

  /**
   * Merge some existing instrumentation with this instrumentation.
   * @param instrumentation some existing instrumentation.
   */
  void merge(Instrumentation instrumentation);

  /**
   * @return the set of valid source line numbers
   */
  Set getSourceLineNumbers();

  /**
   * @return the line coverage rate for the class
   */
  double getLineCoverageRate();

  /**
   * @return the branch coverage rate for the class
   */
  double getBranchCoverageRate();

  /**
   * @return the line coverage rate for particular method
   */
  double getLineCoverageRate(String methodNameAndSignature);

  /**
   * @return the branch coverage rate for a particular method
   */
  double getBranchCoverageRate(String methodNameAndSignature);

  /**
   * @return the source file name.
   */
  String getSourceFileName();

  /**
   * @return the method name and signature of each method found in the
   * class represented by this instrumentation.
   */
  Set getMethodNamesAndSignatures();
}
