/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage;

import com.jcoverage.util.ClassGenHelper;
import com.jcoverage.util.InstructionHelper;
import com.jcoverage.util.InstructionListHelper;
import com.jcoverage.util.MethodGenHelper;

import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;

import org.apache.bcel.classfile.Method;

import org.apache.bcel.generic.ClassGen;
import org.apache.bcel.generic.IfInstruction;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.InstructionList;
import org.apache.bcel.generic.InstructionTargeter;
import org.apache.bcel.generic.InvokeInstruction;
import org.apache.bcel.generic.LDC;
import org.apache.bcel.generic.LineNumberGen;
import org.apache.bcel.generic.MethodGen;
import org.apache.bcel.generic.RET;
import org.apache.bcel.generic.Type;

import org.apache.oro.text.regex.MalformedPatternException;
import org.apache.oro.text.regex.Pattern;
import org.apache.oro.text.regex.Perl5Compiler;
import org.apache.oro.text.regex.Perl5Matcher;

/**
 * Add bytecode instrumentation to the method.
 */
class InstrumentMethodGen {
  final Method original;
  final MethodGenHelper methodGenHelper;
  final ClassGenHelper classGenHelper;

  /**
   * The set of "conditionals" (@see Conditional). Whenever a
   * conditional branch is encountered it is recorded here, including
   * the next Java source line after the conditional branch, and the
   * Java source line of the branch target. This information is later
   * used to calculate the branch coverage rate for this method.
   */
  final Set conditionals=new HashSet();

  /**
   * The set of "valid" source lines. That is, those lines of Java
   * source code that do not represent comments, or other syntax
   * "fluff" (e.g., "} else {"), or those lines that have been ignored
   * because they match the ignore regex.
   */
  // for djUnit
//  final Set sourceLineNumbers=new HashSet();
  final Set sourceLineNumbers=new TreeSet();

  final Perl5Matcher pm=new Perl5Matcher();
  Pattern ignoreRegex=null;

  InstrumentMethodGen(Method original,ClassGen cg,String ignoreRegex) {
	this.original=original;
	this.methodGenHelper=new MethodGenHelper(new MethodGen(original,cg.getClassName(),cg.getConstantPool()));
	this.classGenHelper=ClassGenHelper.newInstance(cg);

	Perl5Compiler pc=new Perl5Compiler();

	if(ignoreRegex!=null) {
	  /**
	   * Compile the ignore regex for later usage
	   */
	  try {
		this.ignoreRegex=pc.compile(ignoreRegex);
	  } catch(MalformedPatternException ex) {
		throw new CoverageRuntimeException(ex);
	  }
	}
  }

  void updateTargeters(InstructionHandle oldTarget,InstructionHandle newTarget,InstructionTargeter targeter) {
	targeter.updateTarget(oldTarget,newTarget);
  }

  void updateTargeters(InstructionHandle oldTarget,InstructionHandle newTarget,InstructionTargeter[] targeters) {
	for(int i=0;i<targeters.length;i++) {
	  updateTargeters(oldTarget,newTarget,targeters[i]);
	}
  }

  /**
   * Inserting coverage instrumentation to a method, inserts
   * additional code into the instrumented class. When this happens we
   * need to adjust any targeters of the original instruction so that
   * they instead target the inserted instrumentation. The
   * instrumentation is inserted immediately prior to
   * <code>oldTarget</code>. Adjusting the targeters to
   * <code>newTarget</code> (the start of where the instrumentation
   * has been added) ensures that the instrumentation is invoked as
   * the original code would have been.
   */
  void updateTargeters(InstructionHandle oldTarget,InstructionHandle newTarget) {
	if(oldTarget.hasTargeters()) {
	  updateTargeters(oldTarget,newTarget,oldTarget.getTargeters());
	}
  }

  /**
   * We currently only ignore those invoke instructions that are from
   * a matching regular regular expression.
   */
  boolean isIgnorable(ClassGenHelper helper,InstructionHandle handle) {
	if(InstructionHelper.isInvokeInstruction(handle)) {
	  return pm.matches(helper.getClassName(handle),ignoreRegex);
	}
	return false;
  }

  boolean hasIgnoreRegex() {
	return ignoreRegex!=null;
  }


  /**
   * We can ignore (for the purposes of instrumentation) any set of
   * instructions which are on our ignore list. Taking the instruction
   * handle of the line number, we iterate over the instructions until
   * we meet the next instruction that has a line number. If we
   * encounter an instruction on our ignore list, then we can ignore
   * (for the purposes of instrumentation) this group of instructions.
   */
  boolean isIgnorable(ClassGenHelper helper,LineNumberGen lng) {
	if(!hasIgnoreRegex()) {
	  return false;
	}

	if(isIgnorable(helper,lng.getInstruction())) {
	  return true;
	}

	InstructionHandle handle=lng.getInstruction().getNext();

	while((handle!=null)&&(!methodGenHelper.hasLineNumber(handle))) {
	  if(isIgnorable(helper,handle)) {
		return true;
	  }

	  handle=handle.getNext();
	}

	return false;
  }

  /**
   * We've found a conditional branch instruction. We need to record
   * the line number immediately after the branch, and the line number
   * of the target of the branch. We can later determine branch
   * coverage rates for this method.
   *
   * @param lng the line number that we found the branch instruction
   * @param ifInstruction the actual <code>if</code> instruction
   */
  void addIfInstruction(LineNumberGen lng,IfInstruction ifInstruction) {

	/**
	 * only add the conditional branch if the target has a line number
	 */
	if(methodGenHelper.getLineNumber(ifInstruction.getTarget())!=0) {
	  conditionals.add(ConditionalFactory.newConditional(lng,methodGenHelper.getLineNumber(ifInstruction.getTarget())));
	}
  }

  void handleIfInstruction(LineNumberGen lng) {
	if(InstructionHelper.isIfInstruction(lng)) {
	  addIfInstruction(lng,(IfInstruction)lng.getInstruction().getInstruction());
	  return;
	}

	InstructionHandle handle=lng.getInstruction().getNext();

	while((handle!=null)&&(!methodGenHelper.hasLineNumber(handle))) {
	  if(InstructionHelper.isIfInstruction(handle)) {
		addIfInstruction(lng,(IfInstruction)handle.getInstruction());
		return;
	  }
	  handle=handle.getNext();
	}
  }

  /**
   * Add coverage instrumentation to the instructions representing a
   * line of Java source code.
   */
  void addInstrumentation(LineNumberGen lng) {

	if(isIgnorable(classGenHelper,lng)) {
	  return;
	}

	/**
	 * If we find a conditional branch instruction in the set of
	 * instructions that represent this line of Java source code,
	 * include them in the set of conditionals, so that we can
	 * calculate the branch coverage rate.
	 */
	handleIfInstruction(lng);

	/**
	 * Add this line of Java code to the list of "valid" source lines
	 * for this method
	 */
	addSourceLine(lng);

	/**
	 * Emit and insert the coverage instrumentation code immediately
	 * prior to the first instruction representing the Java
	 * code. Update any targeters of the original instruction to
	 * instead target the coverage instrumentation code.
	 */
	updateTargeters(lng.getInstruction(),methodGenHelper.getMethodGen().getInstructionList().insert(lng.getInstruction(),emitGetInstrumentationAndTouchLine(lng)));
  }

  /**
   * The core instrumentation. This sequence of instructions is
   * emitted into the instrumented class on every line of original
   * Java code.
   *
   * NOTE THAT THIS EMITTED CODE IS ALSO LICENSED UNDER THE GNU
   * GENERAL PUBLIC LICENSE. NON GPL INSTRUMENTED APPLICATIONS MUST BE
   * LICENSED UNDER SEPARATE AGREEMENT. FOR FURTHER DETAILS, PLEASE
   * VISIT http://jcoverage.com/license.html.
   */
  InstructionList emitGetInstrumentationAndTouchLine(LineNumberGen lng) {
	InstructionList il=new InstructionList();

	/**
	 * Obtain an instance of InstrumentationFactory, via a static call
	 * to InstrumentationFactory.
	 */
	il.append(classGenHelper.createInvokeStatic(InstrumentationFactory.class,"getInstance",InstrumentationFactory.class));

	/**
	 * Create a new instance of Instrumentation (or reuse an existing
	 * instance, if one is already present in the factory), for the
	 * class that we have instrumented.
	 */
	il.append(new LDC(classGenHelper.getConstantPool().addString(classGenHelper.getClassGen().getClassName())));
	il.append(classGenHelper.createInvokeVirtual(InstrumentationFactory.class,"newInstrumentation",Instrumentation.class,String.class));

	/**
	 * Update the coverage counters for this line of source code, by
	 * "touching" its instrumentation.
	 */
	il.append(InstructionListHelper.push(classGenHelper.getConstantPool(),lng.getSourceLine()));
	il.append(classGenHelper.createInvokeInterface(Instrumentation.class,"touch",void.class,int.class));

	return il;
  }

  /**
   * We only record the set of "valid" source lines. That is, source
   * lines that are not comments, or contain other syntax "fluff"
   * (e.g., "} else {"), or any line of code that is being ignored by
   * instrumentation ignore regex. <code>addSourceLine</code> is only
   * called if the source line represented by <code>lng</code> is a
   * "real" line of code.
   */
  void addSourceLine(LineNumberGen lng) {
	sourceLineNumbers.add(new Integer(lng.getSourceLine()));
  }

  void addInstrumentation(LineNumberGen[] lineNumberTable) {
	for(int i=0;i<lineNumberTable.length;i++) {
	  if((i==(lineNumberTable.length-1))&&methodGenHelper.isVoidReturningMethod()&&InstructionHelper.isRetInstruction(lineNumberTable[i])) {
		continue;
	  }

	  addInstrumentation(lineNumberTable[i]);
	}
  }

  /**
   * The entry point for this class. We add coverage instrumentation
   * immediately prior to every instruction found in the line number
   * table.
   */
  public void addInstrumentation() {

	/**
	 * Add instrumentation to this method.
	 */
	addInstrumentation(methodGenHelper.getMethodGen().getLineNumbers());

	/**
	 * Recalculate the maxium stack size necessary for this
	 * instrumented method.
	 */
	methodGenHelper.getMethodGen().setMaxStack();

	/**
	 * Replace the original method, with the instrumented method.
	 */
	classGenHelper.getClassGen().replaceMethod(original,methodGenHelper.getMethodGen().getMethod());
  }

  /**
   * @return the set of valid source line numbers, that is those that
   * are not comments, nor syntax "fluff" (e.g., "} else {"), nor
   * lines that are being ignored by the instrumentation ignore regex.
   */
  Set getSourceLineNumbers() {
	return sourceLineNumbers;
  }

  /**
   * This method is used internally to calculate the branch coverage
   * rate for this method.
   */
  Set getConditionals() {

	return conditionals;
  }
}
