/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage;

import com.jcoverage.util.MethodHelper;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;

import org.apache.bcel.generic.ClassGen;

/**
 * Add coverage instrumentation to an existing class. Instances of
 * this class are normally created by @see Instrument, as part of the
 * instrumentation process.
 */
class InstrumentClassGen {

  final ClassGen cg;

  /**
   * The set of "real" source line numbers that are present in this
   * class. That is, those lines of Java source code that do not
   * represent comments, or other syntax "fluff" (e.g., "} else {"),
   * or those lines that have been ignored because they match the
   * ignore regex.
   */
  final Set sourceLineNumbers=new TreeSet();

  /**
   * The set of method names concatenated with their signature.
   */
  final Set methodNamesAndSignatures=new TreeSet();

  /**
   * A mapping from method name and signature to the set of line
   * numbers for that method.
   */
  final Map methodLineNumbers=new HashMap();

  /**
   * A mapping from method name and signature to the set of
   * conditionals for that method.
   * @see Conditional
   */
  final Map methodConditionals=new HashMap();

  final String ignoreRegex;

  InstrumentClassGen(JavaClass jc,String ignoreRegex) {
	this.cg=new ClassGen(jc);
	this.ignoreRegex=ignoreRegex;
  }

  /**
   * Add instrumentation collected by <code>instrument</code> to this
   * class
   */
  private void add(Method method,InstrumentMethodGen instrument) {
	methodNamesAndSignatures.add(MethodHelper.getMethodNameAndSignature(method));
	methodLineNumbers.put(MethodHelper.getMethodNameAndSignature(method),instrument.getSourceLineNumbers());
	methodConditionals.put(MethodHelper.getMethodNameAndSignature(method),instrument.getConditionals());
	addSourceLineNumbers(instrument.getSourceLineNumbers());
  }

  /**
   * Add instrumentation to a method found in this class.
   * @param method a method present in the class
   */
  void addInstrumentation(Method method) {
	InstrumentMethodGen instrument=new InstrumentMethodGen(method,cg,ignoreRegex);
	instrument.addInstrumentation();
	add(method,instrument);
  }

  /**
   * Add instrument to all the supplied methods.
   */
  void addInstrumentation(Method[] methods) {
	for(int i=0;i<methods.length;i++) {
	  addInstrumentation(methods[i]);
	}
  }

  /**
   * Add coverage instrumentation to the class. Once instrumented, the
   * instrumented class is tagged with a marker interface @see
   * HasBeenInstrumented to prevent it from being instrumented again.
   */
  void addInstrumentation() {

	addInstrumentation(getClassGen().getMethods());
	getClassGen().addInterface(HasBeenInstrumented.class.getName());
  }

  ClassGen getClassGen() {
	return cg;
  }

  private void addSourceLineNumbers(Set sourceLineNumbers) {
	this.sourceLineNumbers.addAll(sourceLineNumbers);
  }

  /**
   * @return the set of source line numbers for this class
   */
  Set getSourceLineNumbers() {
	return sourceLineNumbers;
  }

  /**
   * @return a mapping from method name and signature to the set of
   * line numbers for that method.
   */
  Map getMethodLineNumbers() {
	return methodLineNumbers;
  }

  /**
   * @return a mapping from method name and signature to the set of
   * conditionals for that method.
   * @see Conditional
   */
  Map getMethodConditionals() {
	return methodConditionals;
  }

  /**
   * @return the set of method names and signatures that can be found
   * in this class.
   */
  Set getMethodNamesAndSignatures() {
	return methodNamesAndSignatures;
  }
}
