/**
 * Copyright (C)2004 dGIC Corporation.
 * 
 * This file is part of djUnit plugin.
 * 
 * djUnit plugin is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option) any
 * later version.
 * 
 * djUnit plugin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * jcoverage; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA
 *  
 */
package jp.co.dgic.testing.common;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;

import org.apache.bcel.Constants;
import org.apache.bcel.classfile.Attribute;
import org.apache.bcel.classfile.ConstantPool;
import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.classfile.Unknown;

public class BcelJUnit4AnnotationReader {
	
	private final String ATTR_NAME_ANNOTATION = "RuntimeVisibleAnnotations";
	private static final String JUNIT4_TEST_ANNOTATION = "org/junit/";
	private static final String JUNIT4_TESTRUNNER_ANNOTATION = "org/junit/runner";

	private boolean hasJUnit4Annotation = false;

	public BcelJUnit4AnnotationReader(JavaClass jc) {
		readAnnotations(jc);
	}
	
	public boolean hasJUnit4Annotation() {
		return hasJUnit4Annotation;
	}

	private void readAnnotations(JavaClass jc) {

		Attribute[] attrs = jc.getAttributes();
		checkUnknownAttributes(attrs);
		
		Method[] ms = jc.getMethods();
		for (int i = 0; i < ms.length; i++) {
			if (hasJUnit4Annotation) return;
			checkUnknownAttributes(ms[i].getAttributes());
		}
	}

	private void checkUnknownAttributes(Attribute[] attrs) {
		for (int i = 0; i < attrs.length; i++) {
			if (attrs[i].getTag() == Constants.ATTR_UNKNOWN) {
				Unknown un = (Unknown) attrs[i];
				if (!ATTR_NAME_ANNOTATION.equals(un.getName())) continue;

				DataInputStream in = null;
				try {
					in = new DataInputStream(new ByteArrayInputStream(un.getBytes()));
					short numAnnotations = in.readShort();
					for (int j = 0; j < numAnnotations; j++) {
						if(checkType(in, un.getConstantPool())) return;
					}
				} catch (IOException e) {
					e.printStackTrace();
				} finally {
					if (in != null) {
						try {
							in.close();
						} catch (IOException e) {
						}
					}
				}
			}
		}

	}
	
	private boolean checkType(DataInputStream in, ConstantPool cp) throws IOException {

		short typeIndex = in.readShort();
		String type = cp.constantToString(cp.getConstant(typeIndex));

		if (isJUnit4Annotation(type)) {
			hasJUnit4Annotation = true;
			return true;	// JUnit4 annotation is found.
		}
	
		short numMVPairs = in.readShort();
		for (int i = 0; i < numMVPairs; i++) {
			in.readShort();			// name index
			skipReadValue(in, cp);
		}
	
		return false;	// JUnit4 annotation is not found.
	}

	private void skipReadValue (DataInputStream in, ConstantPool cp)	throws IOException {
		in.readByte();	// tag
		in.readShort();	// value
	}
	
	private boolean isJUnit4Annotation(String type) {
		if (type == null) return false;
		if (type.indexOf(JUNIT4_TEST_ANNOTATION) >= 0) return true;
		if (type.indexOf(JUNIT4_TESTRUNNER_ANNOTATION) >= 0) return true;
		return false;
	}

}
