/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.reporting;

import java.util.*;

/**
 * This class provides a skeletal implementation of a {@link Page}.
 * Subclasses should override the {@link
 * #instantiateLineForCategory} method to return Line
 * implementation instances on demand.
 */
public abstract class AbstractPage implements Page {
  
  String label;
  Report report;
  List categories=new ArrayList();
  Line masterLine;
  Map lineSetsByCategory=new HashMap();

  int state=Report.READY;

  protected AbstractPage(String label) {
    this.label=label;
  }

  Set getLineSetForCategory(LineCategory category) {
    if (state==Report.CLOSED) {
      throw new IllegalStateException("Report is closed, cannot add new lines.");
    }
    if (category==null) {
      throw new IllegalArgumentException("Category cannot be null");
    }
    if (!categories.contains(category)) {
      throw new IllegalArgumentException("This page (type "+getClass().getName()+") cannot handle category: "+category);
    }
    Set set=(Set)lineSetsByCategory.get(category);
    if(set==null) {
      set=new HashSet();
      lineSetsByCategory.put(category,set);
    }
    return set;
  }

  /**
   * @return a new line instance, null if no lines match the given category.
   */
  public Line createLine(LineCategory category) {
    Set set=getLineSetForCategory(category);
    Line line=instantiateLineForCategory(category);
    if (line!=null) {
      line.setCategory(category);
      line.setOwner(this);
      line.setReport(report);
      set.add(line);
    }
    return line;
  }

  public void addLineReference(Line line,LineCategory category) {
    getLineSetForCategory(category).add(line);    
  }

  public Line lookupLineByField(LineCategory category,Column column,Object value) {
    Set lines=(Set)lineSetsByCategory.get(category);
    if (lines==null) {
      return null;
    }
    for(Iterator it=lines.iterator();it.hasNext();) {
      Line line=(Line)it.next();
      Object field=line.getField(column);
      if (field!=null && field.equals(value)) {
        return line;
      }
    }
    return null;
  }

  /**
   * 
   */
  public Set getLines(LineCategory category) {
    Set result=(Set)lineSetsByCategory.get(category);
    return result!=null?result:Collections.EMPTY_SET;

  }

  /**
   * Subclasses should override this method to return {@link
   * Line} implementation instances on demand.
   * @param category guarenteed not to be null
   */
  protected Line instantiateLineForCategory(LineCategory category) {
    return null;
  }

  /**
   * Subclasses should implemented this method to return the
   * categories of {@link Line line}s that are applicable to this
   * report, or else make exclusive use of the {@link #addCategory}
   * method.
   */
  public LineCategory[] getCategories() {
    return (LineCategory[])categories.toArray(new LineCategory[categories.size()]);
  }

  public void addCategory(LineCategory category) {
    categories.add(category);
  }

  public void setMasterLine(Line masterLine) {
    this.masterLine=masterLine;
  }

  public Line getMasterLine() {
    return masterLine;
  }

  public void setReport(Report report) {
    this.report=report;
  }

  /**
   * Call this method to indicate that no further lines will be
   * created for this report and it can be considered immutable from
   * the point-of-view of formatting.
   */
  public void close() throws ReportingException {
    if(state==Report.READY) {
      state=Report.CLOSED;
      // TODO: Ensure report items are closed
      for(Iterator it=lineSetsByCategory.values().iterator();it.hasNext();) {
        close((Set)it.next());
      }
      if (report.getCollator()!=null) {
        report.getCollator().pageClosed(this);
      }
    } else if (state==Report.CLOSED) {
      throw new IllegalStateException("Aleady closed");
    } else {
      throw new IllegalStateException("Page in unknown state: "+state);
    }
  }

  public boolean isClosed() {
    return state==Report.CLOSED;
  }

  public String getLabel() {
    return label;
  }

  void close(Set set) throws ReportingException {
    for(Iterator it=set.iterator();it.hasNext();) {
      Line line=(Line)it.next();
      if(!line.isClosed()) {
        try {
          line.close();
        } catch(ReportingException ex) {
          throw new ReportingException("Failed to close line "+line,ex);
        }
      }
    }
  }
  
}
