/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage.reporting.xml;

import com.jcoverage.coverage.Instrumentation;
import com.jcoverage.util.ClassHelper;

import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.PrintWriter;

import java.util.Iterator;
import java.util.Map;

public class Main {

  final int indentRate=2;
  final PrintWriter pw;

  int indent=0;

  void increaseIndentation() {
    indent+=indentRate;
  }

  void decreaseIndentation() {
    indent-=indentRate;
  }

  void indent() {
    for(int i=0;i<indent;i++) {
      pw.print(' ');
    }
  }
  
  void println(String ln) {
    indent();
    pw.println(ln);
  }

  public static void main(String[] args) throws Exception {

    LongOpt[] longOpts=new LongOpt[3];
    longOpts[0]=new LongOpt("instrumentation",LongOpt.REQUIRED_ARGUMENT,null,'i');
    longOpts[1]=new LongOpt("output",LongOpt.REQUIRED_ARGUMENT,null,'o');
    longOpts[2]=new LongOpt("source",LongOpt.REQUIRED_ARGUMENT,null,'s');

    Getopt g=new Getopt(Main.class.getName(),args,":i:o:s:",longOpts);

    int c;

    File serializedInstrumentation=null;
    File outputDirectory=null;
    File sourceDirectory=null;

    while((c=g.getopt())!=-1) {
      switch(c) {
      case 'i':
        serializedInstrumentation=new File(g.getOptarg());
        break;
      case 'o':
        outputDirectory=new File(g.getOptarg());
        break;
      case 's':
        sourceDirectory=new File(g.getOptarg());
        break;
      }
    }

    new Main(new FileInputStream(serializedInstrumentation),new PrintWriter(new FileWriter(new File(outputDirectory,"coverage.xml"))),sourceDirectory);
  }


  Main(InputStream serializedInstrumentation,PrintWriter pw,File sourceDirectory) throws IOException,ClassNotFoundException {
    this.pw=pw;
    
    try {
      ObjectInputStream objects=new ObjectInputStream(serializedInstrumentation);

      println("<?xml version=\"1.0\"?>");
      if(sourceDirectory==null) {
        println("<coverage>");
      } else {
        println("<coverage src=\""+sourceDirectory+"\">");
      }
      increaseIndentation();
      dump(((Map)objects.readObject()).entrySet().iterator());
      decreaseIndentation();
      println("</coverage>");
    } finally {
      try {
        serializedInstrumentation.close();
      } catch(IOException ex) {
      }

      pw.close();
    }
  }

  void dump(Iterator entySetIterator) {
    while(entySetIterator.hasNext()) {
      dump((Map.Entry)entySetIterator.next());
    }
  }

  void dump(Map.Entry entry) {
    println("<class name=\""+entry.getKey()+"\">");
    increaseIndentation();
    dump((String)entry.getKey(),(Instrumentation)entry.getValue());
    decreaseIndentation();
    println("</class>");
  }

  String getFileName(String className,Instrumentation instrumentation) {
    return ClassHelper.getPackageName(className).replace('.','/')+'/'+instrumentation.getSourceFileName();
  }

  void dumpMethods(Instrumentation instrumentation) {
    Iterator i=instrumentation.getMethodNamesAndSignatures().iterator();
    while(i.hasNext()) {

      String methodNameAndSignature=(String)i.next();
      
      println("<method nameAndSignature=\""+methodNameAndSignature+"\">");
      increaseIndentation();

      try {
        println("<line rate=\""+instrumentation.getLineCoverageRate(methodNameAndSignature)+"\"/>");
        println("<branch rate=\""+instrumentation.getBranchCoverageRate(methodNameAndSignature)+"\"/>");
      } catch(IllegalArgumentException ex) {
      }

      decreaseIndentation();
      println("</method>");
    }
  }

  void dump(String className,Instrumentation instrumentation) {
    println("<file name=\""+getFileName(className,instrumentation)+"\"/>");
    println("<line rate=\""+instrumentation.getLineCoverageRate()+"\"/>");
    println("<branch rate=\""+instrumentation.getBranchCoverageRate()+"\"/>");


    println("<methods>");
    increaseIndentation();
    dumpMethods(instrumentation);
    decreaseIndentation();
    println("</methods>");
    
    StringBuffer sb=new StringBuffer();
    Iterator i=instrumentation.getSourceLineNumbers().iterator();
    while(i.hasNext()) {
      sb.append(i.next());
      if(i.hasNext()) {
        sb.append(", ");
      }
    }
    println("<valid lines=\""+sb.toString()+"\"/>");

    i=instrumentation.getCoverage().entrySet().iterator();
    while(i.hasNext()) {
      Map.Entry entry=(Map.Entry)i.next();
      println("<line number=\""+entry.getKey()+"\" hits=\""+entry.getValue()+"\"/>");
    }
  }
}
