/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage.reporting.html;

import java.io.*;
import java.util.*;

import com.jcoverage.coverage.reporting.collation.JavaFilePage;
import com.jcoverage.reporting.html.HtmlFormatHelper;
import com.jcoverage.reporting.html.Writable;

/**
 *
 */
public class SourceTable implements Writable {

  File sourceFile;
  JavaFilePage page;

  public SourceTable(File sourceFile,JavaFilePage page) {
	this.sourceFile=sourceFile;
	this.page=page;
  }

  public void writeTo(PrintWriter writer) {

	writer.println("<table border=\"0\" cellpadding=\"2\" cellspacing=\"0\" width=\"85%\">");
	writer.print("<tr>");
	writer.print("<th>Line</th>");
	writer.print("<th>Hits</th>");
	writer.print("<th class=\"remainder\">Source</th>");
	writer.println("</tr>");

	int lineNo=1;             // Java source code lines are 1-based

	try {
	  BufferedReader source=new BufferedReader(new InputStreamReader(new FileInputStream(sourceFile)));

	  Iterator coverage=page.getSourceFileLineCoverageSet().iterator();

	  while(coverage.hasNext()) {
		Map.Entry entry=(Map.Entry)coverage.next();
		int nextLineNo=((Integer)entry.getKey()).intValue();

		while (lineNo<nextLineNo) {
		  String line=source.readLine();
		  if (line==null) {
			break;
		  }
		  writeRow(writer,page,lineNo,0,indentSource(line));
		  lineNo++;
		}

		if (lineNo==nextLineNo) {
		  String line=source.readLine();
		  if (line==null) {
			// As above
			break;
		  }
		  writeRow(writer,page,nextLineNo,((Long)entry.getValue()).longValue(),indentSource(line));
		  lineNo++;
		}

	  }

	  while (true) {
		String remainder=source.readLine();
		if (remainder==null) {
		  // We have come to the end of the file.
		  break;
		}
		writeRow(writer,page,lineNo,0,indentSource(remainder));
		lineNo++;
	  }

	} catch (Exception ex) {
	  writer.println("<p>Failed to read java source file: ");
	  writer.println("<pre>");
	  writer.println(ex.toString());
	  writer.println("</pre>");
	}

	writer.println("</table>");
  }

  private void writeRow(PrintWriter writer,JavaFilePage page,int lineNo,long hits,String line) {
	String cssClass=(lineNo%2==0?"yin":"yang");
	boolean validLine=page.getValidSourceLines().contains(new Integer(lineNo));
	if (validLine && hits==0) {
	  cssClass="highlight";
	}

	writer.print("<tr class=\""+cssClass+"\">");

	writeCell(writer,String.valueOf(lineNo),"lineno");

	if (validLine) {
	  writeCell(writer,String.valueOf(hits),"hits");
	} else {
	  writeCell(writer,"&nbsp;","hits");
	}

	writeCell(writer,"<tt>"+line+"</tt>","code");
	writer.println("</tr>");
  }

  void writeCell(PrintWriter writer,String content,String cssClass) {
	writer.print("<td class=\""+cssClass+"\">"+content+"</td>");
  }

  /**
   * @return a String that has leading spaces converted into non-breaking spaces.
   */
  static String indentSource(String line) {
	StringBuffer sb=new StringBuffer(HtmlFormatHelper.replaceCharacterEntities(HtmlFormatHelper.untabify(line,HtmlFormatHelper.TAB_WIDTH)));
	StringBuffer newsb=new StringBuffer();
	int i=0;
	while(i<sb.length() && sb.charAt(i)==' ') {
	  i++;
	  newsb.append("&nbsp;");
	}
	newsb.append(sb.substring(i));

	// For CSS rendering reasons, we always like to have something in the table cell, even if it's just a space.
	if (newsb.length()==0) {
	  newsb.append("&nbsp;");
	}

	return newsb.toString();
  }

}
