/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage.reporting.html;

import java.io.*;

import com.jcoverage.reporting.Format;
import com.jcoverage.reporting.FormattingContext;
import com.jcoverage.reporting.Page;
import com.jcoverage.reporting.ReportingException;
import com.jcoverage.reporting.View;
import com.jcoverage.reporting.ViewFormattingContext;

import com.jcoverage.coverage.reporting.collation.PackageSummaryPage;
import com.jcoverage.coverage.reporting.collation.ReportSummaryPage;
import com.jcoverage.coverage.reporting.collation.JavaFilePage;
import com.jcoverage.coverage.reporting.collation.StaticFileCollator;

/**
 * This class provides the generation of coverage reports in static
 * html.  Multiple views are defined by this class, and used to
 * generate each page.  Each view corresponds to a different line
 * ordering, so that lines can be generated in alphabetical order, or
 * by ascending or descending coverage.
 */
public class MultiViewStaticHtmlFormat implements Format {
  
  public final static View ALPHABETICAL_VIEW=new AlphabeticalView("By Name");
  public final static View COVERAGE_ASCENDING_VIEW=new CoverageView("By Coverage (Most covered first)",CoverageView.ASCENDING);
  public final static View COVERAGE_DESCENDING_VIEW=new CoverageView("By Coverage (Least covered first)",CoverageView.DESCENDING);

  public final static View[] ALL_VIEWS=new View[]{ALPHABETICAL_VIEW,COVERAGE_ASCENDING_VIEW,COVERAGE_DESCENDING_VIEW};

  public void formatPage(ViewFormattingContext ctx,Page page) throws ReportingException {
    throw new IllegalStateException(getClass().getName()+" must be called with a plain FormattingContext");
  }

  /**
   * Implementation proxy.  This implementation delegates to another
   * formatPage implementation depending on the page.
   * @throws ReportingException if the page is not recognized.
   */
  public void formatPage(FormattingContext ctx,Page page) throws ReportingException {
    
    if (page instanceof ReportSummaryPage || page instanceof PackageSummaryPage) {
      
      Format format=null;

      if (page instanceof ReportSummaryPage) {
        format=new ReportSummaryFormat();
      } else if (page instanceof PackageSummaryPage) {
        format=new PackageSummaryFormat();
      }

      ViewFormattingContext viewContext=new ViewFormattingContext(ctx,ALL_VIEWS,ALPHABETICAL_VIEW);
      format.formatPage(viewContext,page);
      viewContext=new ViewFormattingContext(ctx,ALL_VIEWS,COVERAGE_ASCENDING_VIEW);
      format.formatPage(viewContext,page);
      viewContext=new ViewFormattingContext(ctx,ALL_VIEWS,COVERAGE_DESCENDING_VIEW);
      format.formatPage(viewContext,page);

    } else if (page instanceof JavaFilePage) {

      Format format=new JavaFilePageFormat();
      format.formatPage(ctx,page);

    } else {
      throw new ReportingException("Don't know how to format a page of type "+page.getClass());
    }

  }
  
}
