/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage.reporting.collation;

import java.io.*;

import com.jcoverage.coverage.reporting.collation.JavaFilePage;
import com.jcoverage.coverage.reporting.collation.PackageSummaryPage;
import com.jcoverage.coverage.reporting.collation.ReportSummaryPage;
import com.jcoverage.coverage.reporting.html.MultiViewStaticHtmlFormat;
import com.jcoverage.reporting.AbstractCollator;
import com.jcoverage.reporting.FormattingContext;
import com.jcoverage.reporting.MultiViewCollator;
import com.jcoverage.reporting.Page;
import com.jcoverage.reporting.ReportingException;
import com.jcoverage.reporting.View;
import com.jcoverage.reporting.ViewFormattingContext;
import com.jcoverage.reporting.staticgen.StaticView;

/**
 * This class provides an implementation of a collator which
 * constructs filename paths suitable for a statically generated set
 * of files.
 *
 * <p>
 *
 * As it deals with static files, views must be of type {@link
 * com.jcoverage.reporting.staticgen.StaticView}, since the view's
 * filename modifier is used to construct a unique filename.
 */
public class StaticFileCollator extends AbstractCollator implements MultiViewCollator {
  
  String suffix;

  public StaticFileCollator(String suffix) {
    if (suffix.startsWith(".")) {
      this.suffix=suffix;
    } else {
      this.suffix="."+suffix;
    }
  }

  String getFilenameModifier(View view,Page page) {
    String result="";
    if (view!=null) {
      if (view instanceof StaticView) {
        if (!view.equals(MultiViewStaticHtmlFormat.ALPHABETICAL_VIEW)) {
          String modifier=((StaticView)view).getFilenameModifier(page);
          if (modifier!=null) {
            result=modifier+"/"; // put the view in it's own subdirectory
          }
        }
      } else {
        throw new IllegalStateException("Views given to this StaticFileCollator must be of type StaticView, this one is "+view.getClass());
      }
    }
    return result;
  }

  String getUniquePathFromComponents(View view,Page page,String name,String suffix) {
    return getFilenameModifier(view,page)+name+suffix;
  }

  String getPathToPageView(View view,Page page) {

    if (page instanceof ReportSummaryPage) {
      return getUniquePathFromComponents(view,page,"index",suffix);

    } else if (page instanceof PackageSummaryPage) {
      String packageName=(String)page.getMasterLine().getField(ReportSummaryPackageLine.COLUMN_PACKAGE_NAME);
      return getUniquePathFromComponents(view,page,packageName,suffix);

    } else if (page instanceof JavaFilePage) {
      String clzName=(String)page.getMasterLine().getField(JavaFileLine.COLUMN_FILE_NAME);
      return getUniquePathFromComponents(view,page,clzName,suffix);

    }
    return null;
  }

  public String getPathToPage(FormattingContext ctx,Page page) {
    if (ctx instanceof ViewFormattingContext) {
      return getPathToPageView(((ViewFormattingContext)ctx).getCurrentView(),page);
    } else {
      return getPathToPageView(null,page);
    }
  }

  /**
   *
   */
  public String getPathToPage(FormattingContext ctx,Page page,Page from) {
    return getRelativePath(getPathToPage(ctx,from),getPathToPage(ctx,page));
  }

  public String getPathToResource(FormattingContext ctx,String resource,Page from) {
    return getRelativePath(getPathToPage(ctx,from),resource);
  }

  public String getPathToPage(ViewFormattingContext ctx,Page page,Page from,View toView) {
    return getRelativePath(getPathToPage(ctx,from),getPathToPageView(toView,page));
  }

}
