/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage.reporting;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.jcoverage.coverage.Instrumentation;
import com.jcoverage.coverage.reporting.collation.JavaFileLine;
import com.jcoverage.coverage.reporting.collation.JavaFilePage;
import com.jcoverage.coverage.reporting.collation.PackageSummaryPage;
import com.jcoverage.coverage.reporting.collation.ReportImpl;
import com.jcoverage.coverage.reporting.collation.ReportSummaryPackageLine;
import com.jcoverage.coverage.reporting.collation.ReportSummaryPage;
import com.jcoverage.coverage.reporting.collation.StaticFileCollator;
import com.jcoverage.coverage.reporting.html.MultiViewStaticHtmlFormat;
import com.jcoverage.reporting.Collator;
import com.jcoverage.reporting.FileSerializer;
import com.jcoverage.reporting.Format;
import com.jcoverage.reporting.Line;
import com.jcoverage.reporting.Page;
import com.jcoverage.reporting.Report;
import com.jcoverage.reporting.Serializer;
import com.jcoverage.util.ClassHelper;

/**
 * This class take Instrumentation instances and uses them to drive
 * the generation of a report using the <a href="{@docRoot}/com/jcoverage/coverage/reporting/package-summary.html">report framework</a>.
 */
public class ReportDriver {

  File javaSourceDirectory;

  Report report=new ReportImpl();
  Page indexPage;

  public ReportDriver(File javaSourceDirectory) {
	this.javaSourceDirectory=javaSourceDirectory;
	indexPage=report.createFrontPage();
  }

  public synchronized void addInstrumentation(String clzName,Instrumentation instrumentation) {

	if(!isInnerClass(clzName)) {
	  String id=getSourceFileId(clzName,instrumentation);

	  String sourcePath=id.replace('.','/')+".java";


	  String packageName=ClassHelper.getPackageName(id);
	  if (packageName.equals("")) {
		packageName="default";
	  }

	  // Need to add a package line for this
	  Line packageLine=indexPage.lookupLineByField(ReportSummaryPage.CATEGORY_PACKAGE_SUMMARY,ReportSummaryPackageLine.COLUMN_PACKAGE_NAME,packageName);
	  Page packageDetailPage=null;
	  if(packageLine==null) {

		packageLine=indexPage.createLine(ReportSummaryPage.CATEGORY_PACKAGE_SUMMARY);
		packageLine.setField(ReportSummaryPackageLine.COLUMN_PACKAGE_NAME,packageName);
		packageDetailPage=packageLine.openDetailPage();
	  } else {

		packageDetailPage=packageLine.getDetailPage();
	  }

	  // Now add the class line
	  Line javaFileLine=packageDetailPage.lookupLineByField(PackageSummaryPage.CATEGORY_JAVAFILES,JavaFileLine.COLUMN_FILE_NAME,clzName);
	  Page javaFileDetailPage=null;
	  if (javaFileLine==null) {

		javaFileLine=packageDetailPage.createLine(PackageSummaryPage.CATEGORY_JAVAFILES);
		javaFileLine.setField(JavaFileLine.COLUMN_FILE_NAME,clzName);
		javaFileLine.setField(JavaFileLine.COLUMN_PATH,new File(javaSourceDirectory,sourcePath).getAbsolutePath());
		javaFileDetailPage=javaFileLine.openDetailPage();
	  } else {

		javaFileDetailPage=javaFileLine.getDetailPage();
	  }

	  // Add class line to summary
	  indexPage.addLineReference(javaFileLine,PackageSummaryPage.CATEGORY_JAVAFILES);
	  ((JavaFilePage)javaFileDetailPage).addInstrumentation(instrumentation);
	}
  }

  public void generate(File outputDir) throws Exception {
	Collator collator=new StaticFileCollator(".html");
	report.setCollator(collator);
	Format htmlFormat=new MultiViewStaticHtmlFormat();
	Serializer serializer=new FileSerializer(outputDir);
	collator.addOutputter(htmlFormat,serializer);
	indexPage.close();
  }

  public static String getSourceFileId(String clzName,Instrumentation instrumentation) {

	if (isInnerClass(clzName)) {
	  throw new IllegalStateException("Cannot call this method (getSourceFileId) for an inner class");
	}
	String pkgname=ClassHelper.getPackageName(clzName);


	if(instrumentation.getSourceFileName()==null) {
	  return clzName;
	}

	if (pkgname.equals("")) {
	  return stripJavaSuffix(instrumentation.getSourceFileName());
	} else {
	  return pkgname+"."+stripJavaSuffix(instrumentation.getSourceFileName());
	}
  }

  public static String stripJavaSuffix(String s) {
	return s.substring(0,s.length()-".java".length());
  }

  public static boolean isInnerClass(String clzName) {
	return clzName.indexOf("$")!=-1;
  }
}
