/**
 * www.jcoverage.com
 * Copyright (C)2003 jcoverage ltd.
 *
 * This file is part of jcoverage.
 *
 * jcoverage is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * jcoverage is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jcoverage; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package com.jcoverage.coverage;

import com.jcoverage.util.JavaClassHelper;
import com.jcoverage.util.ClassHelper;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;

import java.util.ResourceBundle;

import org.apache.bcel.classfile.JavaClass;

/**
 * Add coverage instrumentation to existing classes.
 */
public class Instrument {

  File destinationDirectory=null;
  String ignoreRegex=null;
  File baseDir=null;

  /**
   * @param fi a file
   * @return true if the specified file has "class" as its extension,
   * false otherwise.
   */
  boolean isClass(File fi) {

    return fi.getName().endsWith(".class");
  }

  /**
   * @param jc a compiled Java class
   * @return true if the specified class implements the interface
   * @link HasBeenInstrumented, otherwise false.
   */
  boolean isAlreadyInstrumented(JavaClass jc) {

    String[] interfaceNames=jc.getInterfaceNames();
    for(int i=0;i<interfaceNames.length;i++) {

      if(interfaceNames[i].equals(HasBeenInstrumented.class.getName())) {
        return true;
      }
    }
    return false;
  }

  /**
   * @param jc a compiled Java class
   * @return true if the class represented by <code>jc</code> is an
   * interface.
   */
  boolean isInterface(JavaClass jc) {
    return !jc.isClass();
  }

  /**
   * Add coverage instrumentation to the specified Java class.
   * @param clazz a Java class file.
   */
  void instrument(File clazz) {

    InputStream is=null;

    try {
      is=new FileInputStream(clazz);
      JavaClass jc=JavaClassHelper.newJavaClass(is,clazz.getName());
      is.close();
      is=null;

      if(isInterface(jc)||isAlreadyInstrumented(jc)) {
        if(destinationDirectory!=null) {
          /**
           * It is not normally necessary to do anything with an
           * interface or class that has already been
           * instrumented. However, if a destination directory has
           * been specified we copy it to the destination directory,
           * so that on subsequent invocations of "ant" the files will
           * be seen as being upto date, and will not require
           * instrumentation.
           */
          File outputDirectory=new File(destinationDirectory,ClassHelper.getPackageName(jc.getClassName()).replace('.','/'));
          outputDirectory.mkdirs();
          jc.dump(new File(outputDirectory,ClassHelper.getBaseName(jc.getClassName())+".class"));
        }
        return;
      }


      InstrumentClassGen instrument=new InstrumentClassGen(jc,ignoreRegex);
      instrument.addInstrumentation();


      if(destinationDirectory==null) {
        instrument.getClassGen().getJavaClass().dump(clazz);
      } else {
        File outputDirectory=new File(destinationDirectory,ClassHelper.getPackageName(jc.getClassName()).replace('.','/'));
        outputDirectory.mkdirs();
        instrument.getClassGen().getJavaClass().dump(new File(outputDirectory,ClassHelper.getBaseName(jc.getClassName())+".class"));
      }

      InstrumentationInternal i=(InstrumentationInternal)InstrumentationFactory.getInstance().newInstrumentation(jc.getClassName());

      if(instrument.getSourceLineNumbers().isEmpty()) {
      }

      i.setSourceLineNumbers(instrument.getSourceLineNumbers());
      i.setSourceFileName(jc.getSourceFileName());
      i.setSourceLineNumbersByMethod(instrument.getMethodLineNumbers());
      i.setConditionalsByMethod(instrument.getMethodConditionals());
      i.setMethodNamesAndSignatures(instrument.getMethodNamesAndSignatures());
    } catch(IOException ex) {
      
      if(is!=null) {
        try {
          is.close();
        } catch(IOException whileClosing) {
        }
      }

      throw new CoverageRuntimeException(ex);
    }
  }

  void addInstrumentation(File fi) {

    if(fi.isDirectory()) {
      File[] contents=fi.listFiles();
      for(int i=0;i<contents.length;i++) {
        addInstrumentation(contents[i]);
      }
    }

    if(isClass(fi)) {
      instrument(fi);
    }
  }

  void addInstrumentation(String arg) {

    if(baseDir==null) {
      addInstrumentation(new File(arg));
    } else {
      addInstrumentation(new File(baseDir,arg));
    }
  }

  void addInstrumentation(String[] args) {
    for(int i=0;i<args.length;i++) {
      if(args[i].equals("-d")) {
        destinationDirectory=new File(args[++i]);
        continue;
      } else if(args[i].equals("-basedir")) {
        baseDir=new File(args[++i]);
        continue;
      } else if(args[i].equals("-ignore")) {
        ignoreRegex=args[++i];
        continue;
      }

      addInstrumentation(args[i]);
    }
  }

  public static void main(String[] args) {
    Instrument instrument=new Instrument();
    instrument.addInstrumentation(args);
  }
}
