/**
 * Copyright (C)2004 dGIC Corporation.
 *
 * This file is part of djUnit plugin.
 *
 * djUnit plugin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * djUnit plugin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with djUnit plugin; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package jp.co.dgic.testing.common.virtualmock;

import jp.co.dgic.testing.common.AbstractBcelImplementer;
import jp.co.dgic.testing.common.util.DJUnitUtil;

import org.apache.bcel.Constants;
import org.apache.bcel.Repository;
import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.generic.ArrayType;
import org.apache.bcel.generic.BranchInstruction;
import org.apache.bcel.generic.ClassGen;
import org.apache.bcel.generic.CodeExceptionGen;
import org.apache.bcel.generic.ConstantPoolGen;
import org.apache.bcel.generic.INSTANCEOF;
import org.apache.bcel.generic.Instruction;
import org.apache.bcel.generic.InstructionConstants;
import org.apache.bcel.generic.InstructionFactory;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.InstructionList;
import org.apache.bcel.generic.InvokeInstruction;
import org.apache.bcel.generic.MethodGen;
import org.apache.bcel.generic.NOP;
import org.apache.bcel.generic.ObjectType;
import org.apache.bcel.generic.POP;
import org.apache.bcel.generic.PUSH;
import org.apache.bcel.generic.Type;

public class BcelAdviceImplementer extends AbstractBcelImplementer {

	private static final String NULL_RETURN_VALUE_CLASS_NAME = MANAGER_PACKAGE_NAME + "NullReturnValue";

	public BcelAdviceImplementer() {
		super("VirtualMockObjects");
	}

	protected byte[] modify(JavaClass jc) throws Exception {
		return modifyClass(jc);
	}

	public byte[] modifyClass(JavaClass jc) throws ClassNotFoundException {

		String className = jc.getClassName();

		if (!VirtualMockUtil.isUseVirtualMock()) { return null; }

		if (!VirtualMockUtil.isInclude(className)) { return null; }

		if (jc.isInterface()) { return null; }

		ClassGen cg = new ClassGen(jc);

		InstructionFactory ifactory = new InstructionFactory(cg, cg.getConstantPool());

		Method[] methods = cg.getMethods();

		InternalMockObjectManager.printConsole("<<<" + methods.length + " methods found. >>>");

		for (int index = 0; index < methods.length; index++) {

			InternalMockObjectManager.printConsole("#################################################################");
			InternalMockObjectManager.printConsole("#################################################################");
			InternalMockObjectManager.printConsole("### " + (methods[index].isAbstract() == true ? "abstract " : "") + className + "#" + methods[index].getName());
			InternalMockObjectManager.printConsole("#################################################################");
			InternalMockObjectManager.printConsole("#################################################################");

			if (methods[index].isAbstract() || methods[index].isNative()) {
				continue;
			}

			MethodGen mg = new MethodGen(methods[index], className, cg.getConstantPool());

			InstructionList il = mg.getInstructionList();

			// modify method call
			modifyMethodCall(cg, mg, ifactory);

			// insertBefore
			if (isConstructor(mg)) {
				InstructionHandle superOrThisHandle = findSuperOrThis(cg, mg, il, mg.getConstantPool());
				if (superOrThisHandle != null) {
					il.append(superOrThisHandle, createInsertClause(ifactory, mg));
				} else {
					il.insert(createInsertClause(ifactory, mg));
				}
			} else {
				il.insert(createInsertClause(ifactory, mg));
			}

			mg.setMaxStack();
			mg.setMaxLocals();

			cg.replaceMethod(methods[index], mg.getMethod());

			il.dispose();

		}

		return cg.getJavaClass().getBytes();

	}

	private void modifyMethodCall(ClassGen cg, MethodGen mg, InstructionFactory ifactory) throws ClassNotFoundException {

		ConstantPoolGen cpg = mg.getConstantPool();

		InstructionList il = mg.getInstructionList();
		InstructionHandle[] handles = il.getInstructionHandles();

		for (int i = 0; i < handles.length; i++) {

			InvokeInstruction ii = null;

			if (!(handles[i].getInstruction() instanceof InvokeInstruction)) {
				continue;
			}

			ii = (InvokeInstruction) handles[i].getInstruction();

			String className = ii.getClassName(cpg);
			String methodName = ii.getMethodName(cpg);
			Type returnType = Type.getReturnType(ii.getSignature(cpg));

			String signature = ii.getSignature(cpg);
			JavaClass jc = findOwnerClass(mg, className, methodName, signature);
			if (jc != null) {
				className = jc.getClassName();
			}

			boolean isStatic = false;

			if (className.startsWith(MANAGER_PACKAGE_NAME)) {
				continue;
			}

			if (ii.getOpcode() == Constants.INVOKESTATIC) {
				isStatic = true;
			}

			if (isConstructor(methodName)) {

				boolean isAfterDup = isAfterDup(i, handles);
				int indexOfNewOpcode = getIndexOfNewOpcode(i, handles);
				if (indexOfNewOpcode < 0) {
					continue;
				}
				Instruction newInstruction = handles[indexOfNewOpcode].getInstruction();
				createConstructorCall(cg, mg, ifactory, il, ii, handles[i], className, methodName, isAfterDup);
				continue;
			}

			createMethodCall(mg, ifactory, il, ii, handles[i], className, methodName, returnType, isStatic);

		}
	}

	private boolean canReplace(String className) {
		if (DJUnitUtil.isDJUnitSystemClass(className)) { return false; }

		if (DJUnitUtil.isDefaultExcludedPath(className)) { return false; }

		if (isOwnSource(className)) { return false; }

		if (isIgnore(className)) { return false; };

		return true;
	}

	private boolean canNewExprReplace(String className) {

		if (DJUnitUtil.isDJUnitSystemClass(className)) { return false; }

		if (DJUnitUtil.isDefaultExcludedPath(className)) { return false; }

		if (isIgnore(className)) {
			if (isOwnSource(className)) return true;
			return false;
		};

		return true;
	}

	private boolean isIgnore(String className) {
		if (!isIgnoreLibrary()) return false;
		return !VirtualMockUtil.isNotIgnore(className);
	}

	private boolean isIgnoreLibrary() {
		String ignoreLibrary = System.getProperty(VirtualMockUtil.VIRTUALMOCK_IGNORE_LIBRARY_KEY);
		if (ignoreLibrary == null || "".equals(ignoreLibrary)) return false;
		if ("false".equalsIgnoreCase(ignoreLibrary)) return false;
		if ("true".equalsIgnoreCase(ignoreLibrary)) return true;
		return false;
	}

	private void createConstructorCall(ClassGen cg, MethodGen mg, InstructionFactory ifactory, InstructionList il,
										InvokeInstruction ii, InstructionHandle ih, String className,
										String methodName, boolean isAfterDup) throws ClassNotFoundException {

		if (className.equals(cg.getSuperclassName())) return;
		if (className.equals(cg.getClassName())) return;

		ConstantPoolGen cpg = ifactory.getConstantPool();

		String signature = ii.getSignature(cpg);

		final boolean isStatic = false;
		final Type returnType = new ObjectType(className);

		InternalMockObjectManager.printConsole("[INVOKE CONSTRUCTOR] : " + className + "#" + methodName + " " + signature + "(args.length = "
										+ ii.getArgumentTypes(cpg).length + ") " + "(" + ii + ") " + "["
										+ ii.getLoadClassType(cpg) + "] " + (isStatic ? " [static]" : ""));

		if (!canNewExprReplace(className)) { return; }

		// insert call before
		InstructionList beforeCall = new InstructionList();

		// copy to local variables from stack
		beforeCall.append(createCopyStackArgsToLocalVariables(mg, isStatic, ii, ifactory));

		// create arguments array
		int varStartIndex = mg.getMaxLocals() + 1;
		beforeCall.append(createCreateArgsArray(mg, ii.getArgumentTypes(cpg), isStatic, varStartIndex, ifactory));

		// call indicateCalledAndGetReturnValueForNewExpr
		beforeCall.append(new PUSH(cpg, makeKey(className, methodName)));
		beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		beforeCall.append(new PUSH(cpg, isOwnSource(className)));

		beforeCall.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "indicateCalledAndGetReturnValueForNewExpr",
												Type.OBJECT, new Type[] {
													Type.STRING, new ArrayType(Type.OBJECT, 1),
													Type.BOOLEAN
												}, Constants.INVOKESTATIC));

		// copy return value to local variables
		beforeCall.append(InstructionFactory.createStore(Type.OBJECT, mg.getMaxLocals()));

		// if (Mock Value != null)
		beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		BranchInstruction ifnull = InstructionFactory.createBranchInstruction(Constants.IFNULL, null);
		beforeCall.append(ifnull);

		// Mock value is not null.

		// collect arguments at called
		if (isOwnSource(className)) {

			// load mock return value to stack
			beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));

			// create arguments array
			int startIndex = mg.getMaxLocals() + 1;
			beforeCall.append(createCreateArgsArray(mg, ii.getArgumentTypes(cpg), isStatic, startIndex, ifactory));

			// call indicateCalled
			beforeCall.append(new PUSH(cpg, makeKey(className, methodName)));
			beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));

			beforeCall.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "indicateCalled", Type.VOID, new Type[] {
				Type.STRING, new ArrayType(Type.OBJECT, 1)
			}, Constants.INVOKESTATIC));

			// store mock return value to local
			beforeCall.append(InstructionFactory.createStore(Type.OBJECT, mg.getMaxLocals()));
		}

		// throw exceptions
		beforeCall.append(createThrowExceptions(mg, ifactory, className, methodName, signature));
		beforeCall.append(createInvokeThrowException(mg, className, methodName, ifactory));

		// check return value type
		beforeCall.append(createCheckReturnValueOfMethodCall(mg, className, methodName, returnType, ifactory));

		// put mock value into stack
		if (isAfterDup) {
			beforeCall.append(new POP());

			// if ReturnValue instanceof NullReturnValue
			beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
			beforeCall.append(new INSTANCEOF(cpg.addClass(new ObjectType(NULL_RETURN_VALUE_CLASS_NAME))));
			BranchInstruction branch = InstructionFactory.createBranchInstruction(Constants.IFNE, null);
			beforeCall.append(branch);

			// IS NOT NullReturnValue object
			beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
			beforeCall.append(ifactory.createCheckCast(new ObjectType(className)));

			// to return
			BranchInstruction gotoreturn = InstructionFactory.createBranchInstruction(Constants.GOTO, null);
			beforeCall.append(gotoreturn);

			// IS NullReturnValue object
			InstructionHandle ifne = beforeCall.append(InstructionConstants.ACONST_NULL);
			branch.setTarget(ifne);

			// end
			InstructionHandle end = beforeCall.append(new NOP());
			gotoreturn.setTarget(end);
		}

		// goto end.
		BranchInstruction gotoEnd = InstructionFactory.createBranchInstruction(Constants.GOTO, null);
		beforeCall.append(gotoEnd);

		// Mock value is null.
		// put arguments into stack
		InstructionHandle notmock = beforeCall.append(createPutArgsIntoStackFromLocalValriables(mg, isStatic, ii, ifactory));
		ifnull.setTarget(notmock);

		// Invoke real constructor.
		il.insert(ii, beforeCall);

		// end
		InstructionHandle end = il.append(ii, new NOP());
		gotoEnd.setTarget(end);
	}

	private void createMethodCall(MethodGen mg, InstructionFactory ifactory, InstructionList il, InvokeInstruction ii,
									InstructionHandle ih, String className, String methodName, Type returnType, boolean isStatic) throws ClassNotFoundException {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		String signature = ii.getSignature(cpg);

		InternalMockObjectManager.printConsole("[INVOKE METHOD] : " + className + "#" + methodName + " " + signature + "(args.length = "
										+ ii.getArgumentTypes(cpg).length + ") " + "(" + ii + ") " + "["
										+ ii.getLoadClassType(cpg) + "] " + (isStatic ? " [static]" : ""));

		if (!canReplace(className)) { return; }

		// insert call before
		InstructionList beforeCall = new InstructionList();

		// copy to local variables from stack
		beforeCall.append(createCopyStackArgsToLocalVariables(mg, isStatic, ii, ifactory));

		// create arguments array
		int varStartIndex = mg.getMaxLocals() + 1;
		beforeCall.append(createCreateArgsArray(mg, ii.getArgumentTypes(cpg), isStatic, varStartIndex, ifactory));

		// call MockObjectmanager.indicateCalledAndGetReturnValue
		beforeCall.append(new PUSH(cpg, makeKey(className, methodName)));
		beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));

		beforeCall.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "indicateCalledAndGetReturnValue", Type.OBJECT,
												new Type[] {
													Type.STRING, new ArrayType(Type.OBJECT, 1)
												}, Constants.INVOKESTATIC));

		// copy mock return value to local variables
		beforeCall.append(InstructionFactory.createStore(Type.OBJECT, mg.getMaxLocals()));

		// if (Mock value != null)
		beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		BranchInstruction ifnull = InstructionFactory.createBranchInstruction(Constants.IFNULL, null);
		beforeCall.append(ifnull);

		// Mock value is not null.

		// throw exceptions
		beforeCall.append(createThrowExceptions(mg, ifactory, className, methodName, signature));
		beforeCall.append(createInvokeThrowException(mg, className, methodName, ifactory));

		// check return value type
		beforeCall.append(createCheckReturnValueOfMethodCall(mg, className, methodName, returnType, ifactory));

		// return type is not void.
		if (ii.produceStack(cpg) > 0) {

			// if ReturnValue instanceof NullReturnValue
			beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
			beforeCall.append(new INSTANCEOF(cpg.addClass(new ObjectType(NULL_RETURN_VALUE_CLASS_NAME))));
			BranchInstruction branch = InstructionFactory.createBranchInstruction(Constants.IFNE, null);
			beforeCall.append(branch);

			// IS NOT NullReturnValue object
			// put return value into stack
			beforeCall.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));

			// is primitive?
			if (isPrimitive(returnType)) {
				// cast to wrapper class and invoke xxxValue method.
				String wrapperType = toWrapperType(returnType);
				beforeCall.append(ifactory.createCheckCast(new ObjectType(wrapperType)));
				beforeCall.append(ifactory.createInvoke(wrapperType, returnType + "Value", returnType, Type.NO_ARGS,
														Constants.INVOKEVIRTUAL));
			} else {
				// is Object.
				if (isArrayType(returnType)) {
					beforeCall.append(ifactory.createCheckCast((ArrayType) returnType));
				} else {
					beforeCall.append(ifactory.createCheckCast(new ObjectType(returnType.toString())));
				}
			}

			// to return
			BranchInstruction gotoreturn = InstructionFactory.createBranchInstruction(Constants.GOTO, null);
			beforeCall.append(gotoreturn);

			// IS NullReturnValue object
			InstructionHandle ifne = beforeCall.append(toZeroInstruction(returnType, cpg));
			branch.setTarget(ifne);

			InstructionHandle end = beforeCall.append(new NOP());
			gotoreturn.setTarget(end);
		}

		// goto end.
		BranchInstruction gotoEnd = InstructionFactory.createBranchInstruction(Constants.GOTO, null);
		beforeCall.append(gotoEnd);

		// Mock value is null.
		// put local variables into stack
		InstructionHandle notmock = beforeCall.append(createPutArgsIntoStackFromLocalValriables(mg, isStatic, ii, ifactory));
		ifnull.setTarget(notmock);

		// Invoke real method.
		il.insert(ii, beforeCall);

		//  end.
		InstructionHandle end = il.append(ii, new NOP());
		gotoEnd.setTarget(end);

	}

	private InstructionList createInsertClause(InstructionFactory ifactory, MethodGen mg) {

		// insertBefore
		InstructionList beforeList = new InstructionList();

		// indicateCalled
		beforeList.append(createInvokeIndicateCalledAndGetReturnValue(mg, ifactory));

		// getReturnValue
		beforeList.append(createReturnValueProcess(mg, ifactory));

		return beforeList;
	}

	private InstructionList createPrintArgument(int index, Type type, InstructionFactory ifactory) {

		InstructionList il = new InstructionList();

		il.append(ifactory.createFieldAccess("java.lang.System", "out", new ObjectType("java.io.PrintStream"),
												Constants.GETSTATIC));
		il.append(InstructionFactory.createLoad(type, index));

		Type argType = type;
		if (!isPrimitive(type)) {
			argType = Type.OBJECT;
		}

		il.append(ifactory.createInvoke("java.io.PrintStream", "println", Type.VOID, new Type[] {
			argType
		}, Constants.INVOKEVIRTUAL));

		return il;
	}

	private Instruction toZeroInstruction(Type returnType, ConstantPoolGen cpg) {

		if (Type.BOOLEAN.equals(returnType)) { return new PUSH(cpg, false).getInstruction(); }
		if (Type.BYTE.equals(returnType)) { return new PUSH(cpg, (byte) 0).getInstruction(); }
		if (Type.CHAR.equals(returnType)) { return new PUSH(cpg, (char) 0).getInstruction(); }
		if (Type.DOUBLE.equals(returnType)) { return new PUSH(cpg, (double) 0).getInstruction(); }
		if (Type.FLOAT.equals(returnType)) { return new PUSH(cpg, (float) 0).getInstruction(); }
		if (Type.INT.equals(returnType)) { return new PUSH(cpg, (int) 0).getInstruction(); }
		if (Type.LONG.equals(returnType)) { return new PUSH(cpg, (long) 0).getInstruction(); }
		if (Type.SHORT.equals(returnType)) { return new PUSH(cpg, (short) 0).getInstruction(); }

		return InstructionConstants.ACONST_NULL;
	}

	private boolean isArrayType(String typeName) {
		if (typeName == null) { return false; }
		return typeName.endsWith("[]");
	}

	private boolean isArrayType(Type type) {
		return type instanceof ArrayType;
	}

	private InstructionList createInvokeIndicateCalledAndGetReturnValue(MethodGen mg, InstructionFactory ifactory) {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		InstructionList il = new InstructionList();

		// create arguments array
		il.append(createCreateArgsArray(mg, mg.getArgumentTypes(), mg.isStatic(), 0, ifactory));

		// call MockObjecManager.indicateCalledAndGetReturnValue
		il.append(new PUSH(cpg, makeKey(mg.getClassName(), mg.getName())));
		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));

		il.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "indicateCalledAndGetReturnValue", Type.OBJECT, new Type[] {
			Type.STRING, new ArrayType(Type.OBJECT, 1)
		}, Constants.INVOKESTATIC));

		// store mock value to local variables
		il.append(InstructionFactory.createStore(Type.OBJECT, mg.getMaxLocals()));

		return il;
	}

	private InstructionList createReturnValueProcess(MethodGen mg, InstructionFactory ifactory) {

		InstructionList il = new InstructionList();

		// if (returnValue == null) --- to if null then
		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		BranchInstruction ifnull = InstructionFactory.createBranchInstruction(Constants.IFNULL, null);
		il.append(ifnull);

		// if (returnValue != null)
		// throw runtime exception or error
		il.append(createThrowExceptions(mg, ifactory, mg.getExceptions()));
		il.append(createInvokeThrowException(mg, ifactory));

		// return value checks
		Type returnType = Type.getReturnType(mg.getSignature());
		il.append(createCheckReturnValue(mg, mg.getClassName(), mg.getName(), returnType, ifactory));

		// return returnValue
		il.append(createReturnValue(mg, ifactory));

		// if null then
		InstructionHandle nop = il.append(new NOP());
		ifnull.setTarget(nop);

		return il;
	}

	private InstructionList createReturnValue(MethodGen mg, InstructionFactory ifactory) {

		InstructionList il = new InstructionList();

		Type returnType = Type.getReturnType(mg.getSignature());

		// void
		if (Type.VOID.equals(returnType)) {
			il.append(InstructionFactory.createReturn(mg.getReturnType()));
			return il;
		}

		// not void
		// if returnvalue instanceof NullReturnValue
		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		il.append(new INSTANCEOF(mg.getConstantPool().addClass(new ObjectType(NULL_RETURN_VALUE_CLASS_NAME))));
		BranchInstruction branch = InstructionFactory.createBranchInstruction(Constants.IFNE, null);
		il.append(branch);

		// IS NOT NullReturnValue object
		if (!isPrimitive(returnType)) {
			// NOT primitive type
			il.append(InstructionFactory.createLoad(mg.getReturnType(), mg.getMaxLocals()));
			if (isArrayType(returnType)) {
				il.append(ifactory.createCheckCast((ArrayType) returnType));
			} else {
				il.append(ifactory.createCheckCast(new ObjectType(returnType.toString())));
			}

		} else {
			// primitive type
			String wrapperType = toWrapperType(returnType);
			String valueMethod = returnType + "Value";

			il.append(InstructionFactory.createLoad(new ObjectType(wrapperType), mg.getMaxLocals()));
			il.append(ifactory.createCheckCast(new ObjectType(wrapperType)));
			il.append(ifactory
								.createInvoke(wrapperType, valueMethod, returnType, Type.NO_ARGS,
												Constants.INVOKEVIRTUAL));
		}

		// to return
		BranchInstruction gotoreturn = InstructionFactory.createBranchInstruction(Constants.GOTO, null);
		il.append(gotoreturn);

		// IS NullReturnValue object
		InstructionHandle ifne = il.append(toZeroInstruction(returnType, mg.getConstantPool()));
		branch.setTarget(ifne);

		// return
		InstructionHandle end = il.append(InstructionFactory.createReturn(mg.getReturnType()));
		gotoreturn.setTarget(end);

		return il;
	}

	private InstructionList createInvokeThrowException(MethodGen mg, InstructionFactory ifactory) {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		InstructionList il = new InstructionList();

		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		il.append(new PUSH(cpg, makeKey(mg.getClassName(), mg.getName())));

		il.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "throwException", Type.VOID, new Type[] {
			Type.OBJECT, Type.STRING
		}, Constants.INVOKESTATIC));

		return il;

	}

	private InstructionList createInvokeThrowException(MethodGen mg, String className, String methodName, InstructionFactory ifactory) {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		InstructionList il = new InstructionList();

		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		il.append(new PUSH(cpg, makeKey(className, methodName)));

		il.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "throwException", Type.VOID, new Type[] {
			Type.OBJECT, Type.STRING
		}, Constants.INVOKESTATIC));

		return il;

	}

	private InstructionList createCheckReturnValue(MethodGen mg, String className, String methodName, Type returnType,
													InstructionFactory ifactory) {

		// constructor - checkReturnTypeIsIgnoreReturnValue
		if (isConstructor(mg)) { return createInvokeCheckReturnTypeIsIgnoreReturnValue(mg, className, methodName,
																						ifactory); }

		// void - checkReturnTypeIsIgnoreOrNullReturnValue
		if (Type.VOID.equals(returnType)) { return createInvokeCheckReturnTypeIsIgnoreOrNullReturnValue(mg, className,
																										methodName,
																										ifactory); }

		// returnValue
		return createCheckReturnValueType(mg, className, methodName, returnType, ifactory);
	}

	private InstructionList createCheckReturnValueOfMethodCall(MethodGen mg, String className, String methodName,
																Type returnType, InstructionFactory ifactory) {

		// constructor - checkReturnTypeIsIgnoreReturnValue
		if (isConstructor(methodName)) { return createInvokeCheckReturnTypeForNewExpr(mg, className, methodName,
																						ifactory); }

		// void - checkReturnTypeIsIgnoreOrNullReturnValue
		if (Type.VOID.equals(returnType)) { return createInvokeCheckReturnTypeIsIgnoreOrNullReturnValue(mg, className,
																										methodName,
																										ifactory); }

		// returnValue
		return createCheckReturnValueType(mg, className, methodName, returnType, ifactory);
	}

	private InstructionList createInvokeCheckReturnTypeIsIgnoreOrNullReturnValue(MethodGen mg, String className,
																					String methodName,
																					InstructionFactory ifactory) {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		InstructionList il = new InstructionList();

		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		il.append(new PUSH(cpg, makeKey(className, methodName)));

		il.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "checkReturnTypeIsIgnoreOrNullReturnValue", Type.VOID,
										new Type[] {
											Type.OBJECT, Type.STRING
										}, Constants.INVOKESTATIC));

		return il;

	}

	private InstructionList createInvokeCheckReturnTypeIsIgnoreReturnValue(MethodGen mg, String className,
																			String methodName,
																			InstructionFactory ifactory) {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		InstructionList il = new InstructionList();

		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		il.append(new PUSH(cpg, makeKey(className, methodName)));

		il.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "checkReturnTypeIsIgnoreReturnValue", Type.VOID,
										new Type[] {
											Type.OBJECT, Type.STRING
										}, Constants.INVOKESTATIC));

		return il;

	}

	private InstructionList createInvokeCheckReturnTypeForNewExpr(MethodGen mg, String className, String methodName,
																	InstructionFactory ifactory) {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		InstructionList il = new InstructionList();

		// if (!(return value instanceof returnType)) {
		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		il.append(new INSTANCEOF(cpg.addClass(new ObjectType(className))));
		BranchInstruction branch = InstructionFactory.createBranchInstruction(Constants.IFNE, null);
		il.append(branch);

		// return value IS NOT return type.
		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		il.append(new PUSH(cpg, makeKey(className, methodName)));

		il.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "checkReturnTypeForNewExpr", Type.VOID, new Type[] {
			Type.OBJECT, Type.STRING
		}, Constants.INVOKESTATIC));

		// }
		branch.setTarget(il.append(new NOP()));

		return il;

	}

	private InstructionList createCheckReturnValueType(MethodGen mg, String className, String methodName,
														Type returnType, InstructionFactory ifactory) {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		InstructionList il = new InstructionList();

		ObjectType returnObjectType = null;
		if (isPrimitive(returnType)) {
			returnObjectType = new ObjectType(toWrapperType(returnType));
		} else {
			returnObjectType = new ObjectType(returnType.toString());
		}

		// if (!(returnValue instanceof RETURN_TYPE)) {
		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));

		if (isArrayType(returnType)) {
			il.append(new INSTANCEOF(cpg.addArrayClass((ArrayType) returnType)));
		} else {
			il.append(new INSTANCEOF(cpg.addClass(returnObjectType)));
		}
		BranchInstruction branch = InstructionFactory.createBranchInstruction(Constants.IFNE, null);
		il.append(branch);

		// incoke checkReturnTypeIsNullReturnValue
		il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
		il.append(new PUSH(cpg, makeKey(className, methodName)));

		il.append(ifactory.createInvoke(MANAGER_CLASS_NAME, "checkReturnTypeIsNullReturnValue", Type.VOID, new Type[] {
			Type.OBJECT, Type.STRING
		}, Constants.INVOKESTATIC));

		// }
		InstructionHandle nop = il.append(new NOP());
		branch.setTarget(nop);

		return il;

	}

	private InstructionHandle getExceptionHandler(MethodGen mg, InstructionHandle ih, String exceptionType) {
		CodeExceptionGen[] handlers = mg.getExceptionHandlers();

		if (handlers == null) return null;
		for (int i = 0; i < handlers.length; i++) {
			if (handlers[i].getCatchType() == null) continue;
			if (containsTarget(handlers[i], ih)) {
				if (handlers[i].getCatchType().toString().equals(exceptionType)) {
					return handlers[i].getHandlerPC();
				}
			}
		}
		return null;
	}

	private boolean containsTarget(CodeExceptionGen ce, InstructionHandle ih) {
		if (ce.getStartPC().getPosition() > ih.getPosition()) return false;
		if (ce.getEndPC().getPosition() < ih.getPosition()) return false;
		return true;
	}

	private InstructionList createThrowExceptions(MethodGen mg, InstructionFactory ifactory, String className,
													String methodName, String signature) throws ClassNotFoundException {

		return createThrowExceptions(mg, ifactory, getExceptions(className, methodName, signature));
	}

	private String[] getExceptions(String className, String methodName, String signature) throws ClassNotFoundException {
		MethodGen mg = getMethod(className, methodName, signature);

		if (mg == null) return null;

		return mg.getExceptions();
	}

	private MethodGen getMethod(String className, String methodName, String signature) throws ClassNotFoundException {
		JavaClass jc = Repository.lookupClass(className);

		MethodGen mg = getMethod(jc, methodName, signature);

		if (mg != null) return mg;

		JavaClass[] interfaces = jc.getInterfaces();
		if (interfaces == null) return null;
		for (int i  = 0; i < interfaces.length; i++) {
			mg = getMethod(interfaces[i], methodName, signature);
			if (mg != null) return mg;
		}

		return null;
	}

	private MethodGen getMethod(JavaClass jc, String methodName, String signature) throws ClassNotFoundException {

		if (jc == null) return null;

		ClassGen cg = new ClassGen(jc);

		Method method = cg.containsMethod(methodName, signature);
		if (method != null) return new MethodGen(method, jc.getClassName(), cg.getConstantPool());

		if ("java.lang.Object".equals(jc.getClassName())) return null;

		return getMethod(jc.getSuperClass(), methodName, signature);
	}

	private InstructionList createThrowExceptions(MethodGen mg, InstructionFactory ifactory, String[] exceptions) {

		ConstantPoolGen cpg = ifactory.getConstantPool();

		InstructionList il = new InstructionList();

		if (exceptions == null || exceptions.length == 0) {
			il.append(new NOP());
			return il;
		}

		for (int i = 0; i < exceptions.length; i++) {

			InternalMockObjectManager.printConsole("exceptions[" + i + "] : " + exceptions[i]);

			// if (returnValue instanceof xxxException)
			il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));

			il.append(new INSTANCEOF(cpg.addClass(new ObjectType(exceptions[i]))));
			BranchInstruction ifeq = InstructionFactory.createBranchInstruction(Constants.IFEQ, null);
			il.append(ifeq);

			// throw (xxxException) returnValue;
			il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals()));
			il.append(ifactory.createCheckCast(new ObjectType(exceptions[i])));
			il.append(InstructionConstants.ATHROW);

			InstructionHandle nop = il.append(new NOP());
			ifeq.setTarget(nop);

		}

		return il;
	}

	private String makeKey(String className, String methodName) {
		return className + "." + methodName;
	}

}