/**
 * Copyright (C)2004 dGIC Corporation.
 *
 * This file is part of djUnit plugin.
 *
 * djUnit plugin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * djUnit plugin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with djUnit plugin; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */
package jp.co.dgic.testing.common;

import jp.co.dgic.testing.common.util.DJUnitUtil;
import jp.co.dgic.testing.common.virtualmock.VirtualMockUtil;

import org.apache.bcel.Constants;
import org.apache.bcel.Repository;
import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.generic.ClassGen;
import org.apache.bcel.generic.ConstantPoolGen;
import org.apache.bcel.generic.InstructionConstants;
import org.apache.bcel.generic.InstructionFactory;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.InstructionList;
import org.apache.bcel.generic.InvokeInstruction;
import org.apache.bcel.generic.MethodGen;
import org.apache.bcel.generic.NOP;
import org.apache.bcel.generic.PUSH;
import org.apache.bcel.generic.Type;

public abstract class AbstractBcelImplementer extends AbstractBcelModifier {

	protected static final String CONSTRUCTOR_METHOD_NAME = "<init>";
	protected static final String MANAGER_PACKAGE_NAME = "jp.co.dgic.testing.common.virtualmock.";
	protected static final String MANAGER_CLASS_NAME = MANAGER_PACKAGE_NAME + "InternalMockObjectManager";

	public AbstractBcelImplementer(String name) {
		super(name);
	}

	protected boolean isConstructor(MethodGen mg) {
		return isConstructor(mg.getName());
	}

	protected boolean isConstructor(String methodName) {
		return CONSTRUCTOR_METHOD_NAME.equals(methodName);
	}

	protected boolean isSuperOrThis(ClassGen cg, MethodGen mg) {

		return isSuperOrThis(cg, mg.getClassName(), mg.getName());
	}

	protected boolean isSuperOrThis(ClassGen cg, String className, String methodName) {

		if (!isConstructor(methodName)) return false;

		String superClassName = cg.getSuperclassName();
		if (superClassName.equals(className)) return true;
		if (cg.getClassName().equals(className)) return true;
		return false;
	}

	protected boolean isAfterDup(int index, InstructionHandle[] handles) {

		if (index == 0) { return false; }

		short op = 0;
		for (int i = index - 1; i >= 0; i--) {
			op = handles[i].getInstruction().getOpcode();
			if (op == Constants.DUP || op == Constants.DUP_X1) { return true; }
			if (op == Constants.NEW) {
				break;
			}
		}
		return false;
	}

	protected boolean isPrimitive(Type type) {

		if (Type.BOOLEAN.equals(type)) { return true; }
		if (Type.BYTE.equals(type)) { return true; }
		if (Type.CHAR.equals(type)) { return true; }
		if (Type.DOUBLE.equals(type)) { return true; }
		if (Type.FLOAT.equals(type)) { return true; }
		if (Type.INT.equals(type)) { return true; }
		if (Type.LONG.equals(type)) { return true; }
		if (Type.SHORT.equals(type)) { return true; }

		return false;
	}

	protected String toWrapperType(Type type) {

		if (Type.BOOLEAN.equals(type)) { return "java.lang.Boolean"; }
		if (Type.BYTE.equals(type)) { return "java.lang.Byte"; }
		if (Type.CHAR.equals(type)) { return "java.lang.Character"; }
		if (Type.DOUBLE.equals(type)) { return "java.lang.Double"; }
		if (Type.FLOAT.equals(type)) { return "java.lang.Float"; }
		if (Type.INT.equals(type)) { return "java.lang.Integer"; }
		if (Type.LONG.equals(type)) { return "java.lang.Long"; }
		if (Type.SHORT.equals(type)) { return "java.lang.Short"; }

		return "java.lang.Object";
	}

	protected boolean isTwoEntryType(Type type) {
		return "long".equals(type.toString()) || "double".equals(type.toString());
	}

	protected JavaClass findOwnerClass(MethodGen mg, String className, String methodName, String signature) throws ClassNotFoundException {

		if (DJUnitUtil.isDJUnitSystemClass(className)) return null;
		if (DJUnitUtil.isDefaultExcludedPath(className)) return null;
//		if (!isSuperClass(mg.getClassName(), className)) return null;

		JavaClass jc = Repository.lookupClass(className);

		return findOwnerClass(jc, methodName, signature);
	}

	protected JavaClass findOwnerClass(JavaClass jc, String methodName, String signature) throws ClassNotFoundException {

		if (jc == null) return null;
		if (!isOwnSource(jc.getClassName())) return jc;

		ClassGen cg = new ClassGen(jc);

		Method method = cg.containsMethod(methodName, signature);

		if (method != null) return jc;

		if ("java.lang.Object".equals(jc.getClassName())) return null;

		return findOwnerClass(jc.getSuperClass(), methodName, signature);
	}

	protected boolean isSuperClass(String className, String superClassName) throws ClassNotFoundException {
		JavaClass jc = Repository.lookupClass(className);
		return jc.getSuperClass().getClassName().equals(superClassName);
	}

	protected boolean isOwnSource(String className) {

		if (VirtualMockUtil.getIncludeValue() != null && VirtualMockUtil.isInclude(className)) { return true; }
		if (DJUnitUtil.isProjectsSource(className)) { return true; }

		return false;
	}

	/**
	 * Create arguments array object.
	 * 
	 * @param mg MethodGen object
	 * @param argTypes argument types
	 * @param isStatic true : static method, false non static method
	 * @param varStartIndex local variable's start index
	 * @param ifactory InstructionFactory object
	 * @return InstructionList
	 */
	protected InstructionList createCreateArgsArray(MethodGen mg, Type[] argTypes, boolean isStatic, int varStartIndex,
													InstructionFactory ifactory) {

		InstructionList il = new InstructionList();

		int argLength = argTypes.length;

		// create array object
		il.append(new PUSH(ifactory.getConstantPool(), argLength));
		il.append(ifactory.createNewArray(Type.OBJECT, (short) 1));
		il.append(InstructionFactory.createStore(Type.OBJECT, mg.getMaxLocals()));

		int varIndex = varStartIndex;
		if (!isStatic) {
			varIndex++;
		}
		Type argType = null;
		String wrapperTypeName = null;
		for (int i = 0; i < argLength; i++) {
			il.append(InstructionFactory.createLoad(Type.OBJECT, mg.getMaxLocals())); // argument array object
			il.append(new PUSH(ifactory.getConstantPool(), i)); // index

			argType = argTypes[i];
			if (isPrimitive(argType)) {
				// primitive
				wrapperTypeName = toWrapperType(argType);
				il.append(ifactory.createNew(wrapperTypeName));
				il.append(InstructionConstants.DUP);
				il.append(InstructionFactory.createLoad(argType, varIndex));
				il.append(ifactory.createInvoke(wrapperTypeName, CONSTRUCTOR_METHOD_NAME, Type.VOID, new Type[] {
					argType
				}, Constants.INVOKESPECIAL));

				if (isTwoEntryType(argType)) {
					varIndex++;
				}
			} else {
				// object
				il.append(InstructionFactory.createLoad(Type.OBJECT, varIndex));
			}
			varIndex++;

			il.append(InstructionConstants.AASTORE);
		}

		return il;
	}

	protected InstructionList createPutArgsIntoStackFromLocalValriables(MethodGen mg, boolean isStatic,
																		InvokeInstruction ii,
																		InstructionFactory ifactory) {

		InstructionList il = new InstructionList();

		int argLength = ii.getArgumentTypes(ifactory.getConstantPool()).length;
		int argIndex = mg.getMaxLocals() + 1;

		if (!isStatic) {
			il.append(InstructionFactory.createLoad(Type.OBJECT, argIndex));
			argIndex++;
		}

		for (int idx = 0; idx < argLength; idx++) {
			il.append(InstructionFactory.createLoad(ii.getArgumentTypes(ifactory.getConstantPool())[idx], argIndex));
			if (isTwoEntryType(ii.getArgumentTypes(ifactory.getConstantPool())[idx])) {
				argIndex++;
			}
			argIndex++;
		}

		if (il.isEmpty()) {
			il.append(new NOP());
		}

		return il;
	}

	protected InstructionList createCopyStackArgsToLocalVariables(MethodGen mg, boolean isStatic, InvokeInstruction ii,
																InstructionFactory ifactory) {

		InstructionList il = new InstructionList();

		int argLength = ii.getArgumentTypes(ifactory.getConstantPool()).length;
		int argIndex = mg.getMaxLocals() + argLength;

		argIndex += getTwoEntryTypeCount(ii.getArgumentTypes(ifactory.getConstantPool()));

		if (!isStatic) {
			argIndex++;
		}

		for (int idx = argLength - 1; idx >= 0; idx--) {
			if (isTwoEntryType(ii.getArgumentTypes(ifactory.getConstantPool())[idx])) {
				argIndex--;
			}
			il.append(InstructionFactory.createStore(ii.getArgumentTypes(ifactory.getConstantPool())[idx], argIndex));
			argIndex--;
		}

		if (!isStatic) {
			il.append(InstructionFactory.createStore(Type.OBJECT, argIndex));
		}

		return il;

	}

	protected int getTwoEntryTypeCount(Type[] argTypes) {
		if (argTypes == null) return 0;
		int count = 0;
		for (int i = 0; i < argTypes.length; i++) {
			if (isTwoEntryType(argTypes[i])) {
				count++;
			}
		}
		return count;
	}

	protected int getIndexOfNewOpcode(int index, InstructionHandle[] handles) {

		if (index == 0) { return -1; }

		InstructionHandle h = null;
		for (int i = index - 1; i >= 0; i--) {
			h = handles[i];
			if (h.getInstruction().getOpcode() == Constants.NEW) { return i; }
		}
		return -1;
	}

	protected InstructionHandle findSuperOrThis(ClassGen cg, MethodGen mg, InstructionList il, ConstantPoolGen cpg) {
		InstructionHandle[] handles = il.getInstructionHandles();
		for (int i = 0; i < handles.length; i++) {
			if (handles[i].getInstruction() instanceof InvokeInstruction) {
				String className =  ((InvokeInstruction) handles[i].getInstruction()).getClassName(cpg);
				String methodName = ((InvokeInstruction) handles[i].getInstruction()).getMethodName(cpg);
				if (!isSuperOrThis(cg, className, methodName)) continue;
				if (handles[i].getInstruction().getOpcode() == Constants.INVOKESPECIAL) { return handles[i]; }
			}
		}
		return null;
	}

}
